/**
 * ╔═══════════════════════════════════════════════════════════════════════════╗
 * ║                    LUXELIFE - LIFE AUDIT SPREADSHEET                      ║
 * ║                           FREE VERSION                                     ║
 * ║                         www.luxesoft.store                                ║
 * ╚═══════════════════════════════════════════════════════════════════════════╝
 * 
 * FREE VERSION FEATURES:
 * ✓ Basic habit heatmap coloring
 * ✓ Manual template setup
 * ✓ Goal tracking framework
 * 
 * UPGRADE TO PRO FOR:
 * ★ Automatic weekly PDF email reports
 * ★ Smart insights ("20% more productive on Tuesdays")
 * ★ Monthly trend analysis
 * ★ Automatic heatmap updates
 * ★ Streak tracking
 * 
 * Upgrade at: https://luxesoft.store/subscribe-luxelife.html
 */

// ============================================================================
// CONFIGURATION
// ============================================================================

const CONFIG_FREE = {
  // Heatmap Colors (Green gradient based on completion %)
  COLORS: {
    EMPTY: '#ffffff',      // White - no data
    LEVEL_0: '#f3f4f6',    // Gray - 0%
    LEVEL_1: '#dcfce7',    // Light green - 1-25%
    LEVEL_2: '#86efac',    // Medium green - 26-50%
    LEVEL_3: '#22c55e',    // Green - 51-75%
    LEVEL_4: '#16a34a',    // Dark green - 76-99%
    LEVEL_5: '#15803d',    // Darkest green - 100%
  }
};

// ============================================================================
// MENU SETUP
// ============================================================================

/**
 * Creates custom menu when spreadsheet opens
 */
function onOpen() {
  const ui = SpreadsheetApp.getUi();
  ui.createMenu('🌱 LuxeLife FREE')
    .addItem('📊 Update Heatmap Colors', 'updateHeatmapFree')
    .addItem('🎯 Create Habit Template', 'createBasicTemplate')
    .addSeparator()
    .addItem('⭐ Upgrade to Pro', 'showUpgradeInfo')
    .addToUi();
}

// ============================================================================
// HEATMAP COLORING (FREE FEATURE)
// ============================================================================

/**
 * Updates heatmap colors based on completion values
 * Select a range of cells with percentages (0-100) before running
 */
function updateHeatmapFree() {
  const sheet = SpreadsheetApp.getActiveSheet();
  const range = sheet.getActiveRange();
  
  if (!range) {
    SpreadsheetApp.getUi().alert('Please select a range of cells first!');
    return;
  }
  
  const values = range.getValues();
  const backgrounds = [];
  
  for (let i = 0; i < values.length; i++) {
    backgrounds[i] = [];
    for (let j = 0; j < values[i].length; j++) {
      const value = values[i][j];
      backgrounds[i][j] = getColorForValue(value);
    }
  }
  
  range.setBackgrounds(backgrounds);
  
  SpreadsheetApp.getUi().alert(
    '✅ Heatmap Updated!\n\n' +
    'Colors applied based on completion percentage.\n\n' +
    '💡 Upgrade to Pro for automatic updates!'
  );
}

/**
 * Returns color based on completion value
 */
function getColorForValue(value) {
  if (value === '' || value === null || value === undefined) {
    return CONFIG_FREE.COLORS.EMPTY;
  }
  
  const num = typeof value === 'number' ? value : parseFloat(value);
  
  if (isNaN(num) || num === 0) return CONFIG_FREE.COLORS.LEVEL_0;
  if (num <= 25) return CONFIG_FREE.COLORS.LEVEL_1;
  if (num <= 50) return CONFIG_FREE.COLORS.LEVEL_2;
  if (num <= 75) return CONFIG_FREE.COLORS.LEVEL_3;
  if (num < 100) return CONFIG_FREE.COLORS.LEVEL_4;
  return CONFIG_FREE.COLORS.LEVEL_5;
}

// ============================================================================
// BASIC TEMPLATE
// ============================================================================

/**
 * Creates a basic habit tracking template
 */
function createBasicTemplate() {
  const ss = SpreadsheetApp.getActiveSpreadsheet();
  
  // Create or get the Habits sheet
  let sheet = ss.getSheetByName('Habits');
  if (!sheet) {
    sheet = ss.insertSheet('Habits');
  } else {
    sheet.clear();
  }
  
  // Set up headers
  const headers = ['Habit', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun', 'Weekly %'];
  sheet.getRange(1, 1, 1, headers.length).setValues([headers]);
  
  // Style headers
  const headerRange = sheet.getRange(1, 1, 1, headers.length);
  headerRange.setBackground('#10b981');
  headerRange.setFontColor('#ffffff');
  headerRange.setFontWeight('bold');
  headerRange.setHorizontalAlignment('center');
  
  // Add sample habits
  const sampleHabits = [
    ['🏃 Exercise', '', '', '', '', '', '', '', ''],
    ['📚 Read 30 mins', '', '', '', '', '', '', '', ''],
    ['🧘 Meditate', '', '', '', '', '', '', '', ''],
    ['💧 Drink 8 glasses water', '', '', '', '', '', '', '', ''],
    ['😴 Sleep 7+ hours', '', '', '', '', '', '', '', '']
  ];
  
  sheet.getRange(2, 1, sampleHabits.length, sampleHabits[0].length).setValues(sampleHabits);
  
  // Add formulas for weekly percentage
  for (let row = 2; row <= 6; row++) {
    sheet.getRange(row, 9).setFormula(`=IF(COUNTA(B${row}:H${row})>0, COUNTIF(B${row}:H${row}, "✓")/7*100, "")`);
  }
  
  // Add instructions
  sheet.getRange(8, 1).setValue('📝 Instructions:');
  sheet.getRange(9, 1).setValue('• Enter ✓ when you complete a habit');
  sheet.getRange(10, 1).setValue('• Select the Weekly % column and run "Update Heatmap Colors" from the menu');
  sheet.getRange(11, 1).setValue('• Upgrade to Pro for automatic weekly email reports!');
  
  // Adjust column widths
  sheet.setColumnWidth(1, 200);
  for (let i = 2; i <= 8; i++) {
    sheet.setColumnWidth(i, 60);
  }
  sheet.setColumnWidth(9, 100);
  
  SpreadsheetApp.getUi().alert(
    '✅ Template Created!\n\n' +
    'Your basic habit tracker is ready.\n\n' +
    '1. Enter ✓ for completed habits\n' +
    '2. Use menu to update heatmap colors\n\n' +
    '⭐ Upgrade to Pro for:\n' +
    '• Automatic weekly email reports\n' +
    '• Smart productivity insights\n' +
    '• Streak tracking'
  );
}

// ============================================================================
// UPGRADE PROMPT
// ============================================================================

/**
 * Shows upgrade information
 */
function showUpgradeInfo() {
  const html = HtmlService.createHtmlOutput(`
    <div style="font-family: Arial, sans-serif; padding: 20px; text-align: center;">
      <h2 style="color: #10b981;">⭐ Upgrade to LuxeLife Pro</h2>
      <p style="font-size: 14px; color: #666;">Unlock powerful automation features:</p>
      <ul style="text-align: left; font-size: 13px;">
        <li><strong>📧 Weekly Email Reports</strong> - PDF summary every Sunday</li>
        <li><strong>🔥 Smart Insights</strong> - "20% more productive on Tuesdays"</li>
        <li><strong>📈 Trend Analysis</strong> - Monthly progress charts</li>
        <li><strong>🎯 Streak Tracking</strong> - Never break the chain</li>
        <li><strong>⚡ Auto Heatmap</strong> - Colors update automatically</li>
      </ul>
      <br>
      <a href="https://luxesoft.store/subscribe-luxelife.html" target="_blank" 
         style="background: #10b981; color: white; padding: 12px 24px; 
                text-decoration: none; border-radius: 8px; font-weight: bold;">
        Upgrade Now - $9.99
      </a>
      <p style="font-size: 11px; color: #999; margin-top: 15px;">One-time payment • Lifetime access</p>
    </div>
  `)
    .setWidth(350)
    .setHeight(350);
  
  SpreadsheetApp.getUi().showModalDialog(html, 'Upgrade to Pro');
}
